// copyright 2008 Neil Barnes 
//
// nailed_barnacle@hotmail.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the license, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// This software modified from existing Fiat Coupe Punto software to 
// talk to Fiat Punto (Mk1) 55 1.1 SPI with the IAW 16F family ECU
// also includes:
// - 500 (899/1000 SPI)
// - Panda (899/1000 SPI, 1108 4*4, 1108 SPI CA)
// - Punto (55 1.1 SPI, 60 1.2 SPI, Selecta 1.2 SPI, 1.1 SPI Em 04 East Europe)
// - Lancia Y (giovani 1.1 SPI, 1.2 SPI CM, 1.2 SPI CA)
// - Palio (1108 SPI em 04 turkey, south africa, morocco)
// - Tipo 1372 SPI TOFAS
// - Tipo/Tempra (1.6 SPI, USA, Em 04)
// - Seicento (899 SPI Base, 899 SPI Fizione Electronica, 1102 SPI Base and Condition)
//
// note - this ECU requires an initialiation sequence at least seven seconds after the
// ECU has powered up, with a fixed delay between initialisation bytes.

// turn off all the MS VC2005 deprecated warnings

#define _CRT_SECURE_NO_DEPRECATE

#include "stdafx.h"


#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <windows.h>
#include <stdio.h>
#include <io.h>
#include <mmsystem.h>
#include <setupapi.h>
#include <objbase.h>
#include <initguid.h>
#include <devguid.h>
#include <regstr.h>
#include <mbstring.h>

// The following define is from ntddser.h in the DDK. It is also
// needed for serial port enumeration.
#ifndef GUID_CLASS_COMPORT
DEFINE_GUID(GUID_CLASS_COMPORT, 0x86e0d1e0L, 0x8089, 0x11d0, 0x9c, 0xe4, \
			0x08, 0x00, 0x3e, 0x30, 0x1f, 0x73);
#endif

#include "emudisplay.h"

static _enginetype et = NOTHING;
static BOOL Logging = FALSE;

// Globals for serial stuff

static HANDLE  rs232;
static DCB			dcb;
static DWORD		txcount,rxcount;
static COMMTIMEOUTS	cto;
static char comport[128];
static int baudrate;

// the tx and rx frame buffers

unsigned char txframe[15];
unsigned char rxframe[15];
unsigned char rxbuffer[64];

////////////////////////////////////////////////////////////////////////////////////////////////////////////
int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpszCmdLine, int nCmdShow)
{   

MSG         msg ;      // a message
WNDCLASS    wndclass ; // the window class 


	if (!hPrevInstance)    // load data into window class struct. 
    {
      wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
      wndclass.lpfnWndProc   = (WNDPROC)WndProc ;
      wndclass.cbClsExtra    = 0 ;
      wndclass.cbWndExtra    = 0 ;
      wndclass.hInstance     = hInstance ;
      wndclass.hIcon         = LoadIcon (hInstance, "IDI_ICON1") ;
			wndclass.hCursor       = LoadCursor (NULL, IDC_ARROW) ;
      wndclass.hbrBackground = (HBRUSH)GetStockObject (BLACK_BRUSH) ;
	  	wndclass.lpszMenuName  = MAKEINTRESOURCE(IDR_MENU1) ;
	  	wndclass.lpszClassName = szProgName ;
          
                                    // register the window class 
      if (!RegisterClass (&wndclass))
      	return FALSE ;
    }
    
	ghInstance = hInstance;

	hWnd = CreateWindowEx (      	// create the program's window here
        0l,
				szProgName,             // class name 
        szProgName,             // window name 
				WS_OVERLAPPEDWINDOW,
				//| WS_VSCROLL,						// window style
        CW_USEDEFAULT,					// x position on screen 
				CW_USEDEFAULT,					// y position on screen
				630,
				585,
        NULL,                   // parent window handle (null = none) 
        NULL,                   // menu handle (null = class menu) 
        hInstance,              // instance handle 
        NULL) ;                 // lpstr (null = not used) 

    ShowWindow (hWnd, nCmdShow) ;   
		UpdateWindow (hWnd) ;       // send first WM_PAINT message

    while (GetMessage (&msg, NULL, 0, 0))   // the message loop 
    {
    	TranslateMessage (&msg) ;
			DispatchMessage (&msg) ;
    }
    return (int)msg.wParam ;
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
Gauge::Gauge (char * _title, char * _units, int _height, int _x, int _y, int _low, int _high, 
							int _majortick, int _minortick,	int _hieek, int _hiwarn, int _lowarn, int _loeek,
							int _places)
{
	// constructor for gauge; it just sets all the parameters
	height = _height;
	x = _x;
	y = _y;
	low = _low;
	high = _high;
	majortick = _majortick;
	minortick = _minortick;
	hieek = _hieek;
	hiwarn = _hiwarn;
	lowarn = _lowarn;
	loeek = _loeek;
	value = (float)low;
	places = _places;
	strcpy(title,_title);
	strcpy(units,_units);
}

void Gauge::Show (HDC hDC, float _value)
{
	// draw the gauge - called in wm_paint

HBRUSH	holdbrush;

	holdbrush = (HBRUSH)SelectObject(hDC,hbMurple);
	RoundRect (hDC,x+1,y+1,x+99,y+height-1,30,30);

	// the main bar
	Rectangle (hDC, x+20,y+25,x+25,y+height-24);

	SelectObject(hDC,holdbrush);

int bigticks;
int littleticks;
char tickmark[10];

POINT arrow[3];

char amount[20];

	value = max(_value,low);
	value = min(_value,high);

	bigticks = (high-low)/majortick;
	littleticks = (high-low)/minortick;

	// ticks
	for (int q = 0; q <= bigticks; q++)
	{
		MoveToEx(hDC,x+25,(y+height-25)-(q*(height-50)/bigticks),NULL);
		LineTo(hDC,x+35,(y+height-25)-(q*(height-50)/bigticks));
		sprintf(tickmark,"%d",((high-low)*q/bigticks)+low);
		SetBkColor(hDC,MURPLE);
		SetTextAlign(hDC,TA_BASELINE);
		TextOut(hDC,x+38,(y+height-20)-(q*(height-50)/bigticks),tickmark,(int)strlen(tickmark));
 	}

	for (int q = 0; q <= (high-low)/minortick; q++)
	{
		MoveToEx(hDC,x+25,(y+height-25)-(q*(height-50)/littleticks),NULL);
		LineTo(hDC,x+30,(y+height-25)-(q*(height-50)/littleticks));
 	}

	// we need to undraw the old pointer and draw a new one
	
	// we load the arrow array with the vertices
	arrow[1].x = x+19;
	arrow[1].y = (y+height-25)-(int)((value-low)*(height-50)/(high-low));
	arrow[2].x = x+9;
	arrow[2].y = arrow[1].y+10;
	arrow[0].x = x+9;
	arrow[0].y = arrow[1].y-10;

	// select the colour of the arrow
	if (value > hieek)
		holdbrush = (HBRUSH)SelectObject(hDC,hbRed);
	else
		if (value > hiwarn)
			holdbrush = (HBRUSH)SelectObject(hDC,hbAmber);
		else
			if (value > lowarn)
				holdbrush = (HBRUSH)SelectObject(hDC,hbGreen);
			else
				if (value > loeek)
					holdbrush = (HBRUSH)SelectObject(hDC,hbAmber);
				else
					holdbrush = (HBRUSH)SelectObject(hDC,hbRed);	
	
	Polygon(hDC,arrow,3);

	SelectObject(hDC,holdbrush);

	// more stuff here; undraw the old arrow, draw the new one with colours
	// depending on the value

	// the title of the gauge
	SetTextAlign(hDC,TA_BASELINE | TA_CENTER);
	TextOut(hDC,x+50,y+15,title,(int)strlen(title));

	// and the actual value
	sprintf(amount,"%.*f %s",places,(double)value,units);
	TextOut(hDC,x+50,y+height-5,amount,(int)strlen(amount));
	
}

void Gauge::Rescale(int _low,						// minimum display value
							int _high,					// maximum display value
							int _majortick,			// interval between labeled index marks
							int _minortick,			// interval between unlabeled index marks
							int _hieek,					// high danger level
							int _hiwarn,				// high warning level
							int _lowarn,				// low warning level (i.e. warn if below this value
							int _loeek)					// low danger level
{
	// change the range of the scale without moving it - does not redisplay
	low = _low;
	high = _high;
	majortick = _majortick;
	minortick = _minortick;
	hieek = _hieek;
	hiwarn = _hiwarn;
	lowarn = _lowarn;
	loeek = _loeek;
}

void Gauge::Update (HDC hDC, float _value)
{
	// update the display by redrawing only the pointer and the value

HBRUSH	holdbrush;

	holdbrush = (HBRUSH)SelectObject(hDC,hbMurple);
	SelectObject(hDC,GetStockObject(NULL_PEN));

		// erase the entire pointer space
	RoundRect (hDC,x+2,y+10,x+21,y+height-10, 10,10);
		// and the value display area
	RoundRect (hDC,x+10,y+height-20,x+90,y+height-1,10,10);

	SelectObject(hDC,holdbrush);
	SelectObject(hDC,GetStockObject(BLACK_PEN));


POINT arrow[3];

char amount[20];

	value = max(_value,low);
	value = min(_value,high);

	// we need to undraw the old pointer and draw a new one
	
	// we load the arrow array with the vertices
	arrow[1].x = x+19;
	arrow[1].y = (y+height-25)-(int)((value-low)*(height-50)/(high-low));
	arrow[2].x = x+9;
	arrow[2].y = arrow[1].y+10;
	arrow[0].x = x+9;
	arrow[0].y = arrow[1].y-10;

	// select the colour of the arrow
	if (value > hieek)
		holdbrush = (HBRUSH)SelectObject(hDC,hbRed);
	else
		if (value > hiwarn)
			holdbrush = (HBRUSH)SelectObject(hDC,hbAmber);
		else
			if (value > lowarn)
				holdbrush = (HBRUSH)SelectObject(hDC,hbGreen);
			else
				if (value > loeek)
					holdbrush = (HBRUSH)SelectObject(hDC,hbAmber);
				else
					holdbrush = (HBRUSH)SelectObject(hDC,hbRed);	
	
	Polygon(hDC,arrow,3);

	SelectObject(hDC,holdbrush);
	// the title of the gauge
	SetTextAlign(hDC,TA_BASELINE | TA_CENTER);
	SetBkMode(hDC,TRANSPARENT);

	// and the actual value
	sprintf(amount,"%.*f %s",places,(double)value,units);
	TextOut(hDC,x+50,y+height-5,amount,(int)strlen(amount));
	
}

int	Gauge::Percentage (void)
{
	// returns current value being displayed as a percentage of the range visible
	return (int)((value-low)*100/(high-low));
}

// let's have some gauges to display. They're global, will get deleted on program close

Gauge rpm ("Revs","rpm",400,10,30,0,8000,1000,200,6500,6000,1500,1000,0);
Gauge airtemp ("Air","C",200,110,30,-55,125,30,5,105,95,-60,-60,0);
Gauge pressure ("Pressure","mb",200,210,30,0,1250,250,50,1200,1200,0,0,0);
Gauge throttle ("Throttle","deg",200,310,30,0,90,30,10,90,90,0,0,1);
Gauge battery ("Volts","V",200,410,30,0,15,3,1,15,14,13,12,2);
Gauge injperiod ("Injector","mS",200,110,230,0,15,15,3,15,15,0,0,3);
Gauge injtime ("Inj. timing","deg",200,210,230,0,720,180,45,720,720,0,0,0);
Gauge advance ("Advance","deg",200,310,230,0,75,15,3,75,75,0,0,0);
Gauge vae ("VAE","%",200,410,230,0,100,20,5,100,100,0,0,0);
Gauge watertemp ("Water","C",400,510,30,-55,125,15,5,105,95,-60,-60,0);

// a structure to keep the results from the engine management	query
static _engine engine;

// and one for the massaged data
static _massage massage;

static FILE * fo = NULL;


/////////////////////////////////////////////////////////////////////////////////////////////////////////////

LRESULT CALLBACK WndProc (HWND hWnd, UINT iMessage, WPARAM wParam, LPARAM lParam)
{

// variables for comms list box
static HWND hComlist;
char kpath[] = "HARDWARE\\DEVICEMAP\\SERIALCOMM";
unsigned char lpData[1024];				// buffer for value data 

// variables for comms detection
HDEVINFO hDevInfo;
SP_DEVINFO_DATA DeviceInfoData;
DWORD i;


GUID *guidDev = (GUID*) &GUID_CLASS_COMPORT;

static int init_flag;						// set if we're initialising

int q;
char ch;

// main program loop
	switch (iMessage)
  {

		//----------------------------------------------------------------------
		case WM_CREATE:

			hbRed = CreateSolidBrush(RED);
			hbGreen = CreateSolidBrush(GREEN);
			hbAmber = CreateSolidBrush(AMBER);
			hbMurple = CreateSolidBrush(MURPLE);

			massage.errcoinp = 0;
			massage.errcoout = 0;
			massage.errcofun = 0;

			init_flag = 1;

			// wake up the comms
			// default is to assume we're attached to a Punto, cos that's what daughter drives :)
			// build a drop-down list for the comms port selection
			
			hComlist = CreateWindowEx(
									0l,
									"COMBOBOX",							// class name 
									NULL,										// window name 
									WS_CHILD
									| WS_VISIBLE
									| CBS_HASSTRINGS
									| CBS_SORT
									| CBS_DROPDOWNLIST,						// window style
									20,											// x position on screen 
									6,											// y position on screen
									190,
									200,
									hWnd,                   // parent window handle (null = none) 
									(HMENU) IDW_SETCOMMS,		// menu handle (null = class menu) 
									ghInstance,             // instance handle 
									NULL) ;                 // lpstr (null = not used) 

			ShowWindow(hComlist, SW_SHOW);			// Display the window 

			// Create a HDEVINFO with all present comms devices.
			hDevInfo = SetupDiGetClassDevs(guidDev,
																		0, // Enumerator
																		0,
																		DIGCF_PRESENT | DIGCF_DEVICEINTERFACE );

			if (hDevInfo == INVALID_HANDLE_VALUE)
			{
				// Insert error handling here.
				MessageBox(NULL,"Sorry, there's a problem identifying serial ports. I have to close.","Error",MB_ICONSTOP);
				return 1;
			}

			// Enumerate through all comms devices in Set.

			DeviceInfoData.cbSize = sizeof(SP_DEVINFO_DATA);
			for (i=0;SetupDiEnumDeviceInfo(hDevInfo,i,&DeviceInfoData);i++)
			{
			DWORD DataT;
			LPTSTR buffer = NULL;
			DWORD buffersize = 0;

				// Call function with null to begin with,
				// then use the returned buffer size
				// to Alloc the buffer. Keep calling until
				// success or an unknown failure.

				while (!SetupDiGetDeviceRegistryProperty(
							hDevInfo,
							&DeviceInfoData,
							//SPDRP_DEVICEDESC,
							SPDRP_FRIENDLYNAME,
							&DataT,
							(PBYTE)buffer,
							buffersize,
							&buffersize))
				{
					if (GetLastError() ==
						ERROR_INSUFFICIENT_BUFFER)
					{
						// Change the buffer size.
						if (buffer) LocalFree(buffer);
						buffer = (LPTSTR)LocalAlloc(LPTR,buffersize);
					}
					else
					{
						// Insert error handling here.
						break;
					}
				}
				
				SendMessage(hComlist, CB_ADDSTRING, 0, (LPARAM)buffer);

				if (buffer) LocalFree(buffer);
			}

			if ( GetLastError()!=NO_ERROR &&
			GetLastError()!=ERROR_NO_MORE_ITEMS )
			{
			// Insert error handling here.
			return 1;
			}

			//  Cleanup
			SetupDiDestroyDeviceInfoList(hDevInfo);
	
			// if there are no com ports found, we need to quit with a friendly message
			if (i == 0)			// i counts the number of ports found
			{
				MessageBox(NULL,"I'm sorry - I can't find any serial or USB ports. Please attach one and restart the program.",
									"Fiat ECU monitor - Port Error",MB_ICONSTOP);
				SendMessage(hWnd,WM_DESTROY,0,0);
			}
			else
			{
				// set the default item; first in list
				SendMessage(hComlist, CB_SETCURSEL, (WPARAM)0,0);

				// and then we fake a message from the listbox to ourselves
				// we claim to have changed the contents and that causes an init of the serial port
				SendMessage(hWnd,WM_COMMAND, MAKEWPARAM(IDW_SETCOMMS, CBN_SELCHANGE),	(LPARAM)hComlist);
			}
			
			// when we initialise, we need to send three bytes at 1200 baud with
			// a 110 ms gap between the data bytes - each takes around 8ms so
			// we start the timer at 120ms, do the init, and change to 500ms
			// for later data grabs
			
			// start a timer running
			SetTimer(hWnd,TICKTOCK,120,NULL);

			break;

		//----------------------------------------------------------------------
		case WM_DESTROY:

			KillTimer(hWnd,TICKTOCK);
			_fcloseall();
			PostQuitMessage (0) ;
      break;

		//----------------------------------------------------------------------
		case WM_TIMER:
			// we get here regularly
			// if we're in init, we need to send three bytes in sequence, and then
			// change the timer to a slower speed
			
			if (init_flag == 0)
			{
				// ask the engine what's going on, man...

				// different for turbo and na, 16 and 20v <sigh>

				if (et == NOTHING)
					break;

				if (et == PUNTO)
				{

					// engine period msb,lsb
					engine.revs.little[1] = GetPuntoEngineStatus(1,engine.revs.little[1]);
					engine.revs.little[0] = GetPuntoEngineStatus(2,engine.revs.little[0]);

					// injector period msb,lsb
					engine.injperiod.little[1] = GetPuntoEngineStatus(3,engine.injperiod.little[1]);
					engine.injperiod.little[0] = GetPuntoEngineStatus(4,engine.injperiod.little[0]);

					// advance
					engine.advance = GetPuntoEngineStatus(5,engine.advance);

					// manifold pressure
					engine.vacuum = GetPuntoEngineStatus(6,engine.vacuum);

					// air and water temp
					engine.airtemp = GetPuntoEngineStatus(7,engine.airtemp);
					engine.watertemp = GetPuntoEngineStatus(8,engine.watertemp);

					// throttle valve angle
					engine.throttle = GetPuntoEngineStatus(9,engine.throttle);

					// battery voltage
					engine.battery = GetPuntoEngineStatus(10,engine.battery);

					// VAE duty cycle (idle control valve)
					engine.vae = GetPuntoEngineStatus(12,engine.vae);

					// there are a couple of other things we might read
					// now we get the temporary errors from ram

					engine.errcoinp = GetPuntoEngineStatus(16,engine.errcoinp);
					engine.errcoout = GetPuntoEngineStatus(17,engine.errcoout);
					engine.errcofun = GetPuntoEngineStatus(18,engine.errcofun);
					engine.secoinp = GetPuntoEngineStatus(43,engine.secoinp);
					engine.secoout = GetPuntoEngineStatus(44,engine.secoout);
					engine.secofun = GetPuntoEngineStatus(45,engine.secofun);

					// now we've got everything handy, we have to massage this to something 
					// a human can understand

					// 15,000,000/engine period = revs per minute
					if (engine.revs.big != 0)
						massage.revs = (float)15000000/engine.revs.big;
					else
						massage.revs = (float)0;

					// 2* injector period = period in us
					massage.injperiod = (float)(engine.injperiod.big*2)/1000;  // we like ms

					// advance/2 = degrees
					massage.advance = (float)engine.advance/2;

					// data * 3 = mmHg
					// mmHg * 1000 / 75 = mbar
					massage.vacuum = (float)engine.vacuum*4;

					// air and water temperature direct value but offset to -40C
					massage.airtemp = (float)engine.airtemp-40;
					massage.watertemp = (float)engine.watertemp-40;

					// throttle valve angle is (value * 0.4234) - 2.9638
					massage.throttle = (float)(((float)engine.throttle*0.4234)-2.9638);

					// battery voltage
					// value * 0.0625 volts
					massage.battery = (float)((float)engine.battery*0.0625);

					// and the last of the data displays - vae position
					// just multiply by 100 and divide by 255 for percentage

					massage.vae = (float)engine.vae;
				}

				// then update display

				/*
				sprintf(title,"%02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x",
								engine.revs.little[1],
								engine.revs.little[0],
								engine.injperiod.little[1],
								engine.injperiod.little[0],
								engine.advance,
								engine.vacuum,
								engine.airtemp,
								engine.watertemp,
								engine.throttle,
								engine.battery,
								engine.injtiming,
								engine.vae);

				SetWindowText(hWnd,title);
				*/

				hDC = GetDC(hWnd);

				rpm.Update(hDC,massage.revs);
				airtemp.Update(hDC,massage.airtemp);
				watertemp.Update(hDC,massage.watertemp);
				injperiod.Update(hDC,massage.injperiod);
				advance.Update(hDC,massage.advance);
				pressure.Update(hDC,massage.vacuum);
				throttle.Update(hDC,massage.throttle);
				battery.Update(hDC,massage.battery);
				vae.Update(hDC,massage.vae);

				// now we check to see whether there's been any change in the error status bytes
				if ((engine.errcoinp != massage.errcoinp) || 
						(engine.errcoout != massage.errcoout) || 
						(engine.errcofun != massage.errcofun))
				{
					
					//MessageBox(hWnd,"Different","",MB_OK);
					// if there has, store the new bytes
					massage.errcoinp = engine.errcoinp;
					massage.errcoout = engine.errcoout;									
					massage.errcofun = engine.errcofun;

					// and display the data
					InvalidateRect(hWnd,NULL,FALSE);
					UpdateWindow(hWnd);
				}

				ReleaseDC(hWnd,hDC);

				// and then last thing - if a file was created, we save the data
				// we save the data both as raw data (after conversion to real numbers) 
				// and as percentages of the range visible on the meters
				if (Logging)
				{
					fprintf(fo,"%f, %f, %f, %f, %f, %f, %f, %f, %f, %d, %d, %d, , %d, %d, %d, %d, %d, %d, %d, %d, %d\n", 
									massage.revs, 
									massage.watertemp,
									massage.airtemp,
									massage.throttle,
									massage.injperiod, 
									massage.advance, 
									massage.battery, 
									massage.vacuum, 
									massage.vae,
									engine.errcoinp, 
									engine.errcoout,
									engine.errcofun,
									rpm.Percentage(),
									watertemp.Percentage(),
									airtemp.Percentage(),
									throttle.Percentage(),
									injtime.Percentage(),
									advance.Percentage(),
									battery.Percentage(),
									pressure.Percentage(),
									vae.Percentage());
				}
			}
			else
			{
				// we're initialising...
				switch (init_flag)
				{
					case 1:			// 0x0f
						ch = (char)0x0f;
						WriteFile(rs232,&ch,1,&txcount,NULL);
						init_flag++;
						break;
					case 2:			// 0xaa
						ch = (char)0xaa;
						WriteFile(rs232,&ch,1,&txcount,NULL);
						init_flag++;
						break;
					case 3:			// 0xcc
						ch = (char)0xcc;
						WriteFile(rs232,&ch,1,&txcount,NULL);
						// on the third send we reset the timer to half a second
						KillTimer(hWnd,TICKTOCK);
						SetTimer(hWnd,TICKTOCK,500,NULL);
						//SetComms(7680);
						SendMessage(hWnd,WM_COMMAND, MAKEWPARAM(IDW_SETCOMMS, CBN_SELCHANGE),	(LPARAM)hComlist);

						init_flag = 0;
						break;
					
					default:
						break;		// we should never get here
				}
			}
			break;


		//----------------------------------------------------------------------
		case WM_PAINT:

			BeginPaint(hWnd,&ps);

			/// output the active data
			rpm.Show(ps.hdc,massage.revs);
			airtemp.Show(ps.hdc,massage.airtemp);
			watertemp.Show(ps.hdc,massage.watertemp);
			injperiod.Show(ps.hdc,massage.injperiod);
			advance.Show(ps.hdc,massage.advance);
			pressure.Show(ps.hdc,massage.vacuum);
			throttle.Show(ps.hdc,massage.throttle);
			battery.Show(ps.hdc,massage.battery);
			vae.Show(ps.hdc,massage.vae);

			// and use the error bits to light 'leds'
			// the errors are in uniram1, 2, 2a, 3, 4, and 4a
			// the fault is flagged in 1, 2, 2a, and 3 (we don't look at them all)
			// and the error type (o/c, s/c etc) in 4 and 4a, if applicable
			
HBRUSH	holdbrush;

			holdbrush = (HBRUSH)SelectObject(ps.hdc,hbMurple);
			RoundRect (ps.hdc,11,431,609,529,20,20);
			SelectObject(ps.hdc,holdbrush);

			SetTextAlign(ps.hdc,TA_BASELINE | TA_LEFT);
			SetBkMode(ps.hdc,TRANSPARENT);
			
			if (engine.errcoinp & 1)		// farfalla
			{
				// Throttle sensor problem
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,21,436,201,456,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoinp & 1)
					TextOut(ps.hdc,31,451,"Throttle sensor S/C",19);
				else
					TextOut(ps.hdc,31,451,"Throttle sensor O/C",19);
			}
			else
				TextOut(ps.hdc,31,451,"Throttle position sensor",25);
			
			if (engine.errcoinp & 2)		// pressione
			{
				// Pressure sensor error
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,21,456,201,476,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoinp & 2)
					TextOut(ps.hdc,31,471,"Pressure sensor S/C",19);
				else
					TextOut(ps.hdc,31,471,"Pressure sensor O/C",19);
			}
			TextOut(ps.hdc,31,471,"Pressure sensor",15);

			if (engine.errcoinp & 4)		// sonda staccato o guasta
			{
				// lambda sensor disconnected or broken
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,21,476,201,496,20,20);
				SelectObject(ps.hdc,holdbrush);
			}
			TextOut(ps.hdc,31,491,"Lambda sensor",13);
			
			if (engine.errcoinp & 8)		// temperatura acqua
			{
				// knock sensor activated
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,21,496,201,516,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoinp & 8)
					TextOut(ps.hdc,31,471,"Water temp sens S/C",19);
				else
					TextOut(ps.hdc,31,471,"Water temp sens O/C",19);
			}
			TextOut(ps.hdc,31,511,"Water temp sensor",17);

			if (engine.errcoinp & 16)
			{
				// water sensor problems
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,221,436,401,456,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoinp & 16)
					TextOut(ps.hdc,231,451,"Air temp sensor S/C",19);
				else
					TextOut(ps.hdc,231,451,"Air temp sensor O/C",19);
			}
			else
				TextOut(ps.hdc,231,451,"Air temp sensor",15);

			if (engine.errcoinp & 32)		// Tensione batteria
			{
				// battery voltage
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,221,456,401,476,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoinp & 32)
					TextOut(ps.hdc,231,451,"Battery volts low",18);
				else
					TextOut(ps.hdc,231,451,"Battery volts high",18);
			}
			else
			TextOut(ps.hdc,231,471,"Battery voltage",15);

			if (engine.errcoout & 1)			// comando iniettore
			{
				// Injector driver
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,221,496,401,516,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoout & 1)
					TextOut(ps.hdc,231,511,"Injector driver S/C",19);
				else
					TextOut(ps.hdc,321,511,"Injector driver O/C",19);
			}
			else
				TextOut(ps.hdc,231,511,"Injector driver",15);

			if (engine.errcoout & 2)			// comando bobino 1
			{
				// spark coil 1
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,421,436,601,456,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoout & 2)
					TextOut(ps.hdc,431,451,"Spark coil 1 S/C",16);
				else
					TextOut(ps.hdc,431,451,"Spark coil 1 O/C",16);
			}
			else
				TextOut(ps.hdc,431,451,"Spark coil 1",12);

			if (engine.errcoout & 4)			// comando bobino 2
			{
				// spark coil 2
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,421,456,601,476,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoout & 4)
					TextOut(ps.hdc,431,471,"Spark coil 2 S/C",16);
				else
					TextOut(ps.hdc,431,471,"Spark coil 2 O/C",16);
			}
			TextOut(ps.hdc,431,471,"Spark coil 2",12);

			if (engine.errcoout & 8)			// comando stepper
			{
				// Stepper motor - ICV?
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,421,476,601,496,20,20);
				SelectObject(ps.hdc,holdbrush);
				if (engine.secoout & 8)
					TextOut(ps.hdc,431,471,"Stepper motor S/C",17);
				else
					TextOut(ps.hdc,431,471,"Stepper motor O/C",17);
			}
			TextOut(ps.hdc,431,491,"Stepper motor",13);

			if (engine.errcofun & 32)		// sensore giri 
			{
				// crank sensor
				holdbrush = (HBRUSH)SelectObject(ps.hdc,hbRed);
				RoundRect (ps.hdc,421,496,601,516,20,20);
			}
			TextOut(ps.hdc,431,511,"Crank sensor",12);


			EndPaint(hWnd,&ps);
			
			break;

		//----------------------------------------------------------------------
		
		case WM_COMMAND:

			// defined child window messages are in the high word of wParam
			// the low word has the window ident
			if (HIWORD(wParam) == CBN_SELCHANGE)
			{
				switch LOWORD(wParam)
				{
					case IDW_SETCOMMS:				// the com port selection combo drop-down list
						// we need to find the newly selected item
						q = (int)SendMessage(hComlist,CB_GETCURSEL,0,0);
						// now q has an index into the currently unused lpData
						SendMessage(hComlist,CB_GETLBTEXT,q,(LPARAM)lpData);
						// we're looking for the phrase 'COMxx' somewhere in the string
						// where xx is the port number
						// to be safe let's make everything upper case for now
						for (q=0; q<strlen((const char *)lpData); q++)
						{
							lpData[q] = toupper(lpData[q]);
						}
						// find the string 'COM'
						
char * sub;
						sub = strstr((char *)lpData, "COM");
						// and if it's a sane sort of thing
						if (strncmp("COM",sub,3) == 0)		// if the first three chars are 'COM' we're probably on the right track
						{
							// actually, we already knew that but what the hell, never hurts to check
							// but we'd actually like (well, it seems *windows* would like) us to have 'Comxx' not 'COMXX'
							// so let's deal with that first...
							sub[1] = 'o';
							sub[2] = 'm';
							// and starting at sub[3] should be a number
							for (q=3; q<strlen(sub); q++)
							{
								// go past the number, end the string at first non-numeric
								if (isdigit(sub[q]) == 0)
								{
									// that's the end of the string we need
									sub[q] = '\0';
									break;
								}
							}
							//strcpy(comport,sub);						// so make it the new port
							sprintf(comport,"\\\\.\\%s",sub);
							CloseHandle(rs232);															// close existing port
							
							// if we're initialising, we start at 1200 baud; otherwise, 7680 as being nearest to
							// the correct 7812.5
							if (init_flag == 1)
								baudrate = 1200;
							else
								baudrate = 7680;
							SetComms(baudrate);
						}
						break;

					default:
						break;
				}
			}
			
			switch (wParam)
			{
				case ID_FILE_EXIT:
					SendMessage(hWnd,WM_DESTROY,0,0);
					break;

				case ID_ENGINE_PUNTO55:
					et = PUNTO;
					SetWindowText(hWnd,"Fiat Punto MK1 55");
					pressure.Rescale(0,1250,250,50,1500,1200,0,0);
					InvalidateRect(hWnd,NULL,FALSE);
					UpdateWindow(hWnd);
					break;




HMENU	hmenubar,hdropdown;

				case ID_LOGGING_ON:
					if (!Logging)
					{
						hmenubar = GetMenu(hWnd);
						hdropdown = GetSubMenu(hmenubar,2);
			
						// open a file
						// we'll use a single file and overwrite it every time we run
						// otherwise we'd end up with dozens of files
						// if you want to keep it, change the name after the program has closed

						if ((fo = fopen("engine details.csv","wt")) == 0)
						{
							// couldn't open the file
							MessageBox(hWnd,"Unable to open the log file...\nno data will be logged","File Error",MB_ICONINFORMATION);
							CheckMenuItem(hdropdown,ID_LOGGING_ON,MF_UNCHECKED | MF_BYCOMMAND);
							CheckMenuItem(hdropdown,ID_LOGGING_OFF,MF_CHECKED | MF_BYCOMMAND);
							Logging = FALSE;
						}
						else																																																						
						{
							fprintf(fo,"revs, water temp, air temp, throttle, injector period, injector timing, advance, volts, pressure, VAE, error 1, error 2, error 3, , revs, water temp, air temp, throttle, injector period, injector timing, advance, volts, pressure, VAE\n");
							CheckMenuItem(hdropdown,ID_LOGGING_ON,MF_CHECKED | MF_BYCOMMAND);
							CheckMenuItem(hdropdown,ID_LOGGING_OFF,MF_UNCHECKED | MF_BYCOMMAND);
							Logging = TRUE;
						}
					}
					break;

				case ID_LOGGING_OFF:
					if (Logging)
					{
						hmenubar = GetMenu(hWnd);
						hdropdown = GetSubMenu(hmenubar,2);
						CheckMenuItem(hdropdown,ID_LOGGING_ON,MF_UNCHECKED | MF_BYCOMMAND);
						CheckMenuItem(hdropdown,ID_LOGGING_OFF,MF_CHECKED | MF_BYCOMMAND);
						Logging = FALSE;
						_fcloseall();
					}
					break;

				default:
					break;
			}
			break;

	    default:            // default windows message processing 
  		return DefWindowProc (hWnd, iMessage, wParam, lParam) ;
  }
  return (0L) ;
}

//---------------------------------------------------------------------------
//
// Serial helper functions
//
//---------------------------------------------------------------------------



void SetComms (int baud)
{

	// set the comms specs
	// we use the comms defined in the global 'comport'

	rs232 = CreateFile (comport,											// we want the serial port
											GENERIC_READ|GENERIC_WRITE,		// for reading and writing
											0,														// only by us
											NULL,													// no child security attributes
				              OPEN_EXISTING,								// because it's the com port
											FILE_ATTRIBUTE_NORMAL,				// we don't want overlapped stuff for single bytes
											NULL);												// no template info allowed

	// check the damn thing opened properly
	// if it didn't, we get a stunningly helpful INVALID_HANDLE_VALUE which
	// GetLastError translates into 'file does not exist' which isn't really
	// what we want a user to see but is the best we have available for now
	// Either way we really ought to check...
	
	if (rs232 == INVALID_HANDLE_VALUE)
	{
	LPVOID lpMsgBuf;
 
		FormatMessage( 
				FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
				NULL,
				GetLastError(),
				MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
				(LPTSTR) &lpMsgBuf,
				0,
				NULL);

		// Display the string.
		MessageBox( NULL, (const char *)lpMsgBuf, "GetLastError", MB_OK|MB_ICONINFORMATION );

		// Free the buffer.
		LocalFree( lpMsgBuf );
		
		// and politely crash
		MessageBox( NULL, "There's an issue with the serial port; I can't find it!\nI'm going to have to shut down now.",
								"Serial port error", MB_OK|MB_ICONSTOP);
		SendMessage(hWnd,WM_DESTROY,0,0);
								
	}

// now, see what it looks like
	GetCommState(rs232,&dcb);

	// that has filled the control block with sensible numbers, but we need to change some
	// specifically, it has set the correct size for the size element

	//dcb.BaudRate							= 7680;									// as near as we can get to the required 7812.5
	dcb.BaudRate							= baud;	
	dcb.fBinary								= 1;
	dcb.fParity								= 0;										// no parity
  dcb.fOutxCtsFlow					= 0;
  dcb.fOutxDsrFlow					= 0;
  dcb.fDtrControl						= DTR_CONTROL_DISABLE; 
  dcb.fDsrSensitivity				= 0;
  dcb.fTXContinueOnXoff			= 1; 
  dcb.fOutX									= 0;
  dcb.fInX									= 0;
  dcb.fErrorChar						= 0; 
  dcb.fNull									= 0;
  dcb.fRtsControl						= RTS_CONTROL_DISABLE; 
  dcb.fAbortOnError					= 0; 
  dcb.ByteSize							= 8;
  dcb.Parity								= 0;
  dcb.StopBits							= 0;										// one stop bit

	SetCommState(rs232,&dcb);

	// let's see what the time-out functions are...
	GetCommTimeouts(rs232,&cto);
	cto.ReadTotalTimeoutConstant = 500;								// read timeout of 500mS
	SetCommTimeouts(rs232,&cto);

}


unsigned char	GetPuntoEngineStatus (unsigned char ch, unsigned char noshow)
{

	// send a character to the engine, wait for the response, and return that response
	// if the loop times out, we return 'noshow'

unsigned char ci;
HBRUSH holdbrush;
HDC hDC;

	//return ch; //<<<<<<<<<<test

	// clear the return in case of timeout
	ci = 0;

	PurgeComm(rs232,PURGE_RXABORT	| PURGE_RXCLEAR);
	// write a little red spot to the screen
	hDC = GetDC(hWnd);
	holdbrush = (HBRUSH)SelectObject(hDC,hbRed);
	RoundRect (hDC,1,1,15,15,15,15);

	// send to port
	WriteFile(rs232,&ch,1,&txcount,NULL);
	SleepEx(10,FALSE);

	// read it back
	rxcount = 0;
	ReadFile(rs232,&ci,1,&rxcount,NULL);
	rxcount = 0;
	ReadFile(rs232,&ci,1,&rxcount,NULL);
	if (rxcount == 0)
		ci = noshow;

	SelectObject(hDC,GetStockObject(BLACK_BRUSH));
	RoundRect(hDC,1,1,15,15,15,15);
	SelectObject(hDC,holdbrush);
	ReleaseDC(hWnd,hDC);

	return (ci);

}


// and the sixteen valve codes we need to use to access them are

#define q16_revsh	0x01
#define q16_revsl	0x02
#define q16_injph	0x03
#define q16_injpl	0x04
#define q16_adv		0x05
#define q16_press	0x06
#define q16_air		0x07
#define q16_water	0x08
#define q16_throttle	0x09
#define q16_volts	0x0a
#define q16_inja	0x0c
#define q16_vae		0x0d
#define q16_uni1	0x0f
#define q16_uni2	0x10
#define q16_uni3	0x11

// we will add a few extra for the useful 20v differences
#define q20_volh	0x12
#define q20_voll	0x13
#define q20_lambda 0x14
#define q20_lambdaint 0x15
#define q20_speed	0x16

// engine parameters
unsigned int e_revs;				// rpm
unsigned int e_injection;			// tens of microseconds
unsigned int e_advance;			// degrees
unsigned int e_pressure;			// millibar
int e_airtemp;			// degrees c
int e_watertemp;			// degrees c
unsigned int e_throttle;			// degrees
unsigned int e_battery;			// millivolts
unsigned int e_injectora;			// degrees
unsigned int e_icr;				// percent
unsigned char e_uniram1;			// binary 8 bit
unsigned char e_uniram2;			// binary 8 bit
unsigned char e_uniram3;			// binary 8 bit
unsigned int e_lambda;				// millivolts
unsigned int e_lambdaint;			// percent -108/+108%
unsigned int e_speed;				// vehicle speed in km/h



