// startrek_bosch_M2_10_4.cpp : Defines the entry point for the console application.
//
// a console-only version of startrek, for experiment
//
// usage: startrek [port#]
//
// this program speaks to the fiat coupe 20v and 20vt (not VIS) and it
// assumes the use of a 409.1 KKL type adaptor (i.e. commands are 
// sent to the K line and echoed on the L line; returned values
// are on the L line.)
// it does not try and enumerate a list of ports; it tries to open 
// the serial port specified in the command line and if it cannot
// open then the programme quits
//
// Released under LGPL licence; see www.gnu.org for details
// (c) neil barnes/nailed_barnacle 2009
// nailed_barnacle@hotmail.com
//
// compiles under Visual Studio VC++ 2005

// first we do away with MS non-standard warning crap
// if I write unsafe code you may assume that I want to

#define _CRT_SECURE_NO_DEPRECATE

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <windows.h>

// global variables... how bad is that?
// these are for the serial code
static HANDLE					rs232;
static DCB						dcb;
static DWORD					txcount,rxcount;
static COMMTIMEOUTS		cto;

// some colour definitions
#define RED (FOREGROUND_RED | FOREGROUND_INTENSITY)
#define GREEN (FOREGROUND_GREEN | FOREGROUND_INTENSITY)
#define BLUE (FOREGROUND_BLUE | FOREGROUND_INTENSITY)
#define WHITE (FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE | FOREGROUND_INTENSITY)
#define YELLOW (FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_INTENSITY)


HANDLE serial_setcomms (int baud, char * comport);
void PrintColour(HANDLE con, char * str, int x, int y, WORD colour);

void initbosch (HANDLE);
int getboschID ();
int boschtx (unsigned char ch, int respond);
int boschrx (char * pch, int respond);
int frametx (unsigned char title, unsigned char length);
int framerx (void);

// temperature conversion look up table
int temperature[256] = {
	-33,-33,-32,-31,-31,-30,-30,-29,-29,-28,-28,-27,-26,-25,-25,-24,
	-23,-23,-22,-22,-21,-20,-20,-19,-18,-18,-17,-16,-16,-15,-14,-13,
	-13,-12,-11,-11,-10,-10, -9, -8, -8, -7, -6, -6, -6, -5, -4, -3,
	 -3, -2, -1,  0,  0,  1,  2,  2,  3,  4,  4,  5,  6,  7,  7,  8,
	  9,  9, 10, 10, 11, 11, 12, 12, 13, 14, 14, 15, 16, 17, 18, 18,
	 19, 19, 20, 20, 21, 22, 22, 23, 24, 25, 25, 26, 26, 27, 27, 28,
	 29, 29, 30, 30, 31, 32, 32, 33, 34, 35, 35, 35, 36, 37, 38, 38,
	 39, 39, 40, 41, 41, 42, 43, 43, 44, 45, 45, 46, 47, 47, 48, 48,
	 49, 50, 50, 51, 52, 53, 53, 54, 55, 55, 56, 57, 57, 58, 59, 59,
	 60, 61, 61, 62, 63, 63, 64, 65, 66, 66, 67, 68, 68, 69, 70, 71,
	 71, 71, 72, 73, 73, 74, 75, 75, 75, 76, 77, 77, 78, 78, 79, 80,
	 80, 81, 81, 82, 82, 83, 84, 85, 86, 86, 87, 88, 88, 89, 89, 90,
	 91, 91, 92, 93, 93, 94, 94, 95, 96, 96, 96, 97, 98, 99, 99,100,
	101,101,102,103,103,104,105,106,107,107,108,109,109,110,110,111,
	111,111,112,112,113,114,115,115,115,116,117,124,124,124,124,124,
	124,124,124,124,125,125,125,125,125,125,125,125,125,125,125,125 };

// this defines the frame used to tx and rx data
char isoframe[16];
	
#define isolength 0
#define isocount 1
#define isotitle 2
#define isoinfo 3

#define isoaddress	0x01	
//	request reading of ram address
#define isoactuator	0x04
//	request actuator test
#define isoerrordel	0x05
//	request deletion of error memory
#define isoend	0x06
//	request end of diagnosis
#define isoerrorread 0x07
//	request error memory read
#define isoadread 0x08
//	request reading of AD channels
#define isoack	0x09
//	acknowledge
#define isonack	0x0a
//	no acknowledge
#define isosecurity	0x24
//	transmission of security code
#define isosecreply	0xea
//	response: security code
#define isoadreply 0xfb
//	response: AD channels
#define isoerrorreply 0xfc
//	response: error memory
#define isoramreply	0xfe
//	response: ram data

// the static frame counter
int isoframecount;

#define ETX 0x03

unsigned char ecudata[24];

// the error structure
struct error_t {
	unsigned char e_code;
	unsigned char e_type;
	unsigned char e_env1;
	unsigned char e_env2;
	unsigned char e_count;
	};
	
// this is where we store the errors
error_t faults[10];


////////////////////////////////////////////////////////////////////////////////////

int main (int argc, char * argv[])
{
char tmp[128];		// temporary store for short messages
char kb;					// last keyboard hit
	
float fecu;				// floating point converted value
int tumbler;			// state for the \|/- tumbler

HANDLE con;				// the pointer to the active console
COORD	posn;				// coords for cursor

int error;				// error return from the packet handler
int q;						// generic counter
int loop;					// loop counter for slower graphics update
									
	// start by getting a handle to the console
	con = CreateFile(	"CONOUT$",
										GENERIC_READ | GENERIC_WRITE,
										0,
										NULL,
										OPEN_EXISTING,
										FILE_SHARE_WRITE,
										NULL);

	// label it
	SetConsoleTitle("Star Trek console version 1.0b - Bosch 2.10.4");
	
	// clear the screen (we probably opened from cmd)
	for (int q=0; q<25; q++)
		PrintColour(con,"                                                                               ",0,q,WHITE);
	PrintColour(con,"",0,0,WHITE);
	
	if (argc < 2)
	{
		printf("Usage: startrek [com_number]\n");
		printf("You didn't specify the com port, so assuming com1\n");
	}
	SetConsoleTextAttribute(con,WHITE);
	printf("Star Trek console version 1.01b\n");
	printf("(c) neil barnes/nailed_barnacle 2010\n");
	
	
	// first we try and open the requested port, or we do nothing...
	// we must convert the port - presented as a number - to the expected
	// text format \\.\comxx
	if (argc>1)
		sprintf(tmp,"\\\\.\\com%s",argv[1]);
	else
		sprintf(tmp,"\\\\.\\com1");
	
	// the serial port works at 4800 baud
	printf("Attempting to open the comms port '%s'...\n",tmp);
	if (serial_setcomms (4800,tmp) == 0)
	{
		printf("Unable to open serial port; quiting.\n");
		exit(1);
	}
	else
		printf("Successfully opened port!\n\n");
	// wait a little while and then erase the message
	SleepEx(2000, false);	
	posn.X = 0;
	posn.Y = 2;
	SetConsoleCursorPosition(con,posn);
	printf("																						                      \n");
	printf("                                                                  \n");
	printf("                                                                  \n");
	PrintColour(con,"Hit 'q' to quit",0,3,YELLOW);

	// we need to initialise the system with a 5-baud byte
	// we have to have woken up the comms port first!	
	PrintColour(con,"",0,4,BLUE);
	initbosch (rs232);
	PrintColour(con,"                                ",0,4,BLUE);
	// now we expect the ECU ID data to arrive - we need
	// to keep it since one byte requires acknowledgement
	ReadFile(rs232,tmp,6,&rxcount,NULL);							// expecting six bytes there
	PrintColour(con,"ECU ID: ",0,21,GREEN);
	printf("%0.2X-%0.2X-%0.2X-%0.2X-%0.2X-%0.2X",(unsigned char)tmp[0],(unsigned char)tmp[1],
																					(unsigned char)tmp[2],(unsigned char)tmp[3],
																					(unsigned char)tmp[4],(unsigned char)tmp[5]);
	// we must acknowledge the ECU by returning the inverse of the third byte received
	tmp[2] = ~tmp[2];
	//printf("sent %0.2x ",tmp[2]);
	WriteFile(rs232,&tmp[2],1,&txcount,NULL);
	// grab the echo byte
	ReadFile(rs232,&tmp,1,&rxcount,NULL);
	//printf("read %0.2x ",*tmp);
	
	
	// now the ECU sends us thirty bytes of ID block data
	// it will be wrapped in a number of packets, but we don't know how many
	
	//printf("\nreading ID block\n");
	getboschID();
	
	// now we sit in a loop querying the ECU, receiving the result, 
	// converting it to something human readable, and displaying it.
	// it's slow to get a single byte from the Bosch ECU, so we'll
	// grab a whole handful of them in one hit and then just
	// offset into that array to refer to them
	
	// loop forever
	tumbler = 0;
	loop = 0;
	while (1)
	{
		if (_kbhit())
		{
			kb = _getch();
			switch (kb)
			{
				// quit on q key, either case
				case 'q':
				case 'Q':
					break;
				
				// we can drop in here any other keypresses
				// we might find useful later here
				case 'd':
				case 'D':
					// delete errors
					error = frametx(isoerrordel,0);		// payload zero bytes; ask for error memory read
					error = framerx();					// wait for the reply
					break;
				
				// and then the main loop	
				default:
					break;
			}
		}
		else
		{
			if ((kb == 'q') | (kb == 'Q'))
				break;
			
			// we implement a rotating whirly thing so we can see it's working
			switch (tumbler)
			{
				case 0:
					PrintColour(con,"\\",16,3,YELLOW);
					break;
				case 1:
					PrintColour(con,"|",16,3,YELLOW);
					break;
				case 2:
					PrintColour(con,"/",16,3,YELLOW);
					break;
				case 3:
					PrintColour(con,"-",16,3,YELLOW);
					break;
				default:
					break;
			}
			tumbler ++;
			tumbler &= 3;
			
			// we ask for all the data in two chunks, to save all the
			// overhead of the packet structure
			// first we build the isoframe
			isoframe[isoinfo] = 10;						// ten frames please
			isoframe[isoinfo+1] = 0x20;				// address 0x2000
			isoframe[isoinfo+2] = 0x00;				// address 0x2000
			error = frametx(isoaddress,3);		// payload three bytes; ask for data
			if (error)
				printf("Error in ram address frame tx\n");
				
			error = framerx();								// wait for the reply
			for (q=0; q<10; q++)							// copy return into our store
				ecudata[q] = isoframe[isoinfo+q];
				
			// now grab the few remaining bytes
			isoframe[isoinfo] = 7;						// seven frames please
			isoframe[isoinfo+1] = 0x20;				// address 0x2000
			isoframe[isoinfo+2] = 0x10;				// address 0x2100
			error = frametx(isoaddress,3);		// payload three bytes; ask for data
			if (error)
				printf("Error in ram address frame tx\n");

			error = framerx();								// wait for the reply
			for (q=0; q<7; q++)								// copy return into our store
				ecudata[q+16] = isoframe[isoinfo+q];
			
			// now we have the data in the same order and address offsets as the ECU has them
			// let's see what they look like
			
			// engine period msb,lsb
			// multiply [0] by 40 and if less than 2500 get better resolution:
			// multiply [1] by 10
			fecu = (float)ecudata[0]*40;
			if (fecu < 2500.0)
				fecu = (float)ecudata[1]*10;
			sprintf(tmp,"Revs:\t\t%d rpm     ",(int)fecu);
			if ((fecu < 500) || (fecu > 6500))
				PrintColour(con,tmp,0,5,RED);
			else
				if ((fecu < 800) || (fecu > 6000))
					PrintColour(con,tmp,0,5,YELLOW);
				else
					PrintColour(con,tmp,0,5,GREEN);
			
			// water temp - read from temperature[]
			fecu = (float)temperature[ecudata[2]];
			sprintf(tmp,"Water temp:\t%d degrees    ",(int)fecu);
			if (fecu > 125.0)
				PrintColour(con,tmp,40,5,RED);
			else
				if (fecu > 105.0)
					PrintColour(con,tmp,40,5,YELLOW);
				else
					PrintColour(con,tmp,40,5,GREEN);
					
			// air temp - read from temperature[]
			fecu = (float)temperature[ecudata[3]];
			sprintf(tmp,"Air temp:\t%d degrees    ",(int)fecu);
			if (fecu > 125.0)
				PrintColour(con,tmp,0,6,RED);
			else
				if (fecu > 105.0)
					PrintColour(con,tmp,0,6,YELLOW);
				else
					PrintColour(con,tmp,0,6,GREEN);
					
			// throttle position - datum*96/230.4
			fecu = ((float)ecudata[4])*(float)96.0/(float)230.4;
			sprintf(tmp,"Throttle:\t%d degrees    ",(int)fecu);
			PrintColour(con,tmp,40,6,GREEN);
			
			// mass air flow
			// [5]*1.6, if less than 51, [6]*0.2
			fecu = (float)ecudata[5]*(float)1.6;
			if (fecu<51.0)
				fecu = (float)ecudata[6]*(float)0.2;
			sprintf(tmp,"Mass air flow:\t%d m^3/hr    ",(int)fecu);
			PrintColour(con,tmp,0,7,GREEN);
					
			// advance
			// 78-[7]*.75
			fecu = (float)78-((float)ecudata[7]*(float)0.75);
			sprintf(tmp,"Ign. advance:\t%d degrees    ",(int)fecu);
			PrintColour(con,tmp,40,7,GREEN);
			
			// injection time
			// [8]*.384; if less than 1, [9]/255 
			fecu = (float)ecudata[8]*(float)0.384;
			if (fecu < 1.0)
				fecu = (float)ecudata[9]/(float)255;
			sprintf(tmp,"Injector time:\t%0.1f ms   ",fecu);
			PrintColour(con,tmp,0,8,GREEN);
			
			// battery voltage
			// [16]*0.0705
			fecu = (float)ecudata[16]*(float)0.0705;
			sprintf(tmp,"Battery:\t%2.1f V   ",fecu);
			if ((fecu < 12.0) || (fecu > 14.5))
				PrintColour(con,tmp,40,8,RED);
			else
				if ((fecu < 13.0) || (fecu > 14.0))
					PrintColour (con,tmp,40,8,YELLOW);
				else
					PrintColour (con,tmp,40,8,YELLOW);
			
			// speed
			// [17]*1
			fecu = (float)ecudata[17];
			sprintf(tmp,"Speed:\t\t%d kph   ",(int)fecu);
			PrintColour (con,tmp,0,9,GREEN);
			
			// lambda sensor
			// ([20]*1.25)/255
			fecu = ((float)ecudata[20]*(float)1.25)/(float)255;
			sprintf(tmp,"Lambda:\t\t%1.2f V   ",fecu);
			PrintColour (con,tmp,40,9,GREEN);
			
			// integrated lambda
			// ([22]-128)*0.8
			fecu = ((float)ecudata[22]-(float)128.0)*(float)0.8;
			sprintf(tmp,"Int. lambda:\t%d %%   ",(int)fecu);
			PrintColour(con,tmp,0,10,GREEN);
			
			// now we can get the error data
			// but only every now and then to save time

			loop++;
			if (loop == 20)
			{
				loop = 0;

				// five blocks of five bytes each
				// well that's what the book says - there are at least seven...
				// byte 1 = error code
				// byte 2 = error type
				// byte 3,4 = engine conditions
				// byte 5 = error count
				// the errors are stored in 'faults[x]' where x = 0 to 4
				
				// first we clear the errors
				for (q=0; q<10; q++)
				{
					faults[q].e_type = 0;
					faults[q].e_code = 0;
					faults[q].e_count = 0;
					faults[q].e_env1 = 0;
					faults[q].e_env2 = 0;
				}
				
				// we build an isoframe
				error = frametx(isoerrorread,0);		// payload zero bytes; ask for error memory read
				error = framerx();					// wait for the reply
				// assume it got back in one piece

				q=0;
				while ((isoframe[isotitle] & 0xff) != isoack) //errorreply)
				{
					// ok, it wasn't an ack, so there's some error messages we need to grab
					if (isoframe[isolength] < 8)
						// then it isn't what we expected, probably an ack
						break;

					if (isoframe[isolength] == 8)
					{
						// one error packet
						//printf("\none error packet");
						// copy the packet to the next available slot in the error table
						faults[q].e_code = isoframe[isoinfo];
						faults[q].e_type = isoframe[isoinfo+1];
						faults[q].e_env1 = isoframe[isoinfo+2];
						faults[q].e_env2 = isoframe[isoinfo+3];
						faults[q].e_count = isoframe[isoinfo+4];
						q++;
					}
					else
					{
						// must be two packets to grab
						//printf("\ntwo error packet");
						faults[q].e_code = isoframe[isoinfo];
						faults[q].e_type = isoframe[isoinfo+1];
						faults[q].e_env1 = isoframe[isoinfo+2];
						faults[q].e_env2 = isoframe[isoinfo+3];
						faults[q].e_count = isoframe[isoinfo+4];
						q++;
						faults[q].e_code = isoframe[isoinfo+5];
						faults[q].e_type = isoframe[isoinfo+6];
						faults[q].e_env1 = isoframe[isoinfo+7];
						faults[q].e_env2 = isoframe[isoinfo+8];
						faults[q].e_count = isoframe[isoinfo+9];
						q++;
					}

					// having dealt with the errors, are there any more?
					error = frametx(isoack,0);
					if (error)
						printf("Error in error memory frame internal ack\n\r");
					error = frametx(isoerrorread,0);		// payload zero bytes; ask for error memory read
					error = framerx();					// wait for the reply
				}
				
				//printf("q=%d",q);
				//exit(1);
				// then we print them
				for (q=0; q<7; q++)
				{
				/*	sprintf(tmp,"Fault %d: %0.2x %0.2x %0.2x %0.2x %0.2x", q, faults[q].e_code, 
																																		faults[q].e_type,
																																		faults[q].e_env1,
																																		faults[q].e_env2,
																																		faults[q].e_count);
				*/
					// decode the messages
					switch (faults[q].e_code)
					{
						case 0x01:
							sprintf(tmp,"(%d) ECU internal RAM error", faults[q].e_count);
							break;
							
						case 0x02:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Idle actuator (closing) s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Idle actuator (closing) s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Idle actuator (closing) open circuit", faults[q].e_count);
							break;
									
						case 0x03:
							sprintf(tmp,"(%d) Fuel pump relay s/c to 12v", faults[q].e_count);
							break;
							
						case 0x04:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Idle actuator (opening) s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Idle actuator (opening) s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Idle actuator (opening) open circuit", faults[q].e_count);
							break;
									
						case 0x05:
							sprintf(tmp,"(%d) Crank sensor open cicuit", faults[q].e_count);
							break;
							
						case 0x06:
							sprintf(tmp,"(%d) ECU internal ROM/EPROM error", faults[q].e_count);
							break;
							
						case 0x07:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Air flow meter s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Air flow meter s/c to ground or o/c", faults[q].e_count);
							break;
							
						case 0x08:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Cam sensor s/c to 12v or o/c", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Cam sensor s/c to ground", faults[q].e_count);
							break;
						
						case 0x09:
							sprintf(tmp,"(%d) Speed signal not plausible", faults[q].e_count);
							break;
							
						case 0x0a:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Lambda integrator FR upper limit", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Lambda integrator FR lower limit", faults[q].e_count);
							break;
							
						case 0x0c:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Throttle position sensor past max (253)", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Throttle position sensor past min (5)", faults[q].e_count);
							break;

						case 0x0d:
							sprintf(tmp,"Automatic gearbox input s/c to ground", faults[q].e_count);
							break;
							
						case 0x10:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Modular manifold command s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Modular manifold command s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Modular manifold command open circuit", faults[q].e_count);
							break;
									
						case 0x16:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Timing variator s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Timing variator s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Timing variator open circuit", faults[q].e_count);
							break;
							
						case 0x18:
							sprintf(tmp,"(%d) Cooling fan signal not plausible", faults[q].e_count);
							break;
							
						case 0x1c:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Lambda sensor s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Lambda sensor s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Lambda sensor open circuit", faults[q].e_count);
							break;
									
						case 0x22:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Purge canister s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Purge canister s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Purge canister open circuit", faults[q].e_count);
							break;
									
						case 0x25:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Battery voltage over maximum", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Battery voltage under minimum", faults[q].e_count);
							break;
						
						case 0x2c:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Air temperature sensor s/c to 12v or o/c", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Air temperature sensor s/c to ground", faults[q].e_count);
							break;
									
						case 0x2d:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Water temperature sensor s/c to 12v or o/c", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Water temperature sensor s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x08)
										sprintf(tmp,"(%d) Water temperature sensor implausible", faults[q].e_count);
							break;
									
						case 0x37:
							sprintf(tmp,"(%d) Crank sensor signal implausible", faults[q].e_count);
							break;
							
						case 0x50:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Waste gate s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Waste gate s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Waste gate open circuit", faults[q].e_count);
							break;
									
						case 0x66:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) FRA self-adaptation parameters beyond upper limit", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) FRA self-adaptation parameters beyond lower limit", faults[q].e_count);
							break;
							
						case 0x67:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) DTV self-adaptation parameters beyond upper limit", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) DTV self-adaptation parameters beyond lower limit", faults[q].e_count);
							break;
							
						case 0x68:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) TRA self-adaptation parameters beyond upper limit", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) TRA self-adaptation parameters beyond lower limit", faults[q].e_count);
							break;
							
						case 0x80:
							sprintf(tmp,"Knock sensor 1 signal not plausible", faults[q].e_count);
							break;
							
						case 0x81:
							sprintf(tmp,"Knock sensor 2 signal not plausible", faults[q].e_count);
							break;

						case 0x85:
							sprintf(tmp,"(%d) Supercharging pressure at top limit", faults[q].e_count);
							break;
							
						case 0x89:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Pressure sensor above maximum", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Pressure sensor below minimum", faults[q].e_count);
							break;
						
						case 0x8a:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Supercharging adjustment with throttle closed at top limit", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Supercharging adjustment with throttle closed at bottom limit", faults[q].e_count);
							break;

						case 0x8f:
							sprintf(tmp,"Knock evaluation circuit signal not plausible", faults[q].e_count);
							break;

						case 0xc8:
							sprintf(tmp,"Error between immobiliser and ECU", faults[q].e_count);
						case 0xe1:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Injector 1 s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Injector 1 s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Injector 1 open circuit", faults[q].e_count);
							break;
									
						case 0xe2:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Injector 2 s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Injector 2 s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Injector 2 open circuit", faults[q].e_count);
							break;
									
						case 0xe3:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Injector 3 s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Injector 3 s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Injector 3 open circuit", faults[q].e_count);
							break;
									
						case 0xe4:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Injector 4 s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Injector 4 s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Injector 4 open circuit", faults[q].e_count);
							break;
									
						case 0xe5:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Injector 5 s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Injector 5 s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Injector 5 open circuit", faults[q].e_count);
							break;
									
						case 0xff:
							if (faults[q].e_type & 0x01)
								sprintf(tmp,"(%d) Exhaust gas recirculator s/c to 12v", faults[q].e_count);
							else 
								if (faults[q].e_type & 0x02)
									sprintf(tmp,"(%d) Exhaust gas recirculator s/c to ground", faults[q].e_count);
								else
									if (faults[q].e_type & 0x04)
										sprintf(tmp,"(%d) Exhaust gas recirculator open circuit", faults[q].e_count);
							break;
							
						default:
							sprintf(tmp,"                                                                    ");
							break;
					}
					if (faults[q].e_count < 64)
						PrintColour(con, tmp, 0, 13+q, YELLOW);
					else
						PrintColour(con, tmp, 0, 13+q, RED);
				}
			}
		}	
	}
	
	// finally, tidy up the serial port and the display
	if (rs232 != NULL)
		CloseHandle(rs232);
		
	PrintColour(con,"Bye...\n",0,22,WHITE);
	CloseHandle(con);
	
	// and quit	
	return (0);
}

// provide basic serial system
// including the USB virtual comm port, if any (tested on FTDI232R)

HANDLE serial_setcomms (int baud, char * comport)
{
	// set the comms specs
	if ((rs232 = CreateFile (comport,											// we want the serial port
													GENERIC_READ|GENERIC_WRITE,		// for reading and writing
													0,														// only by us
													NULL,													// no child security attributes
													OPEN_EXISTING,								// because it's the com port
													FILE_ATTRIBUTE_NORMAL,				// we don't want overlapped stuff for single bytes
													NULL))												// no template info allowed
													!= INVALID_HANDLE_VALUE)
	{
		// ok, we opened it!
		// now, see what it looks like
		GetCommState(rs232,&dcb);

		// that has filled the control block with sensible numbers, but we need to change some
		// specifically, it has set the correct size for the size element

		dcb.BaudRate							= baud;	
		dcb.fBinary								= 1;
		dcb.fParity								= 0;										// no parity
		dcb.fOutxCtsFlow					= 0;
		dcb.fOutxDsrFlow					= 0;
		dcb.fDtrControl						= DTR_CONTROL_DISABLE; 
		dcb.fDsrSensitivity				= 0;
		dcb.fTXContinueOnXoff			= 1; 
		dcb.fOutX									= 0;
		dcb.fInX									= 0;
		dcb.fErrorChar						= 0; 
		dcb.fNull									= 0;
		dcb.fRtsControl						= RTS_CONTROL_DISABLE; 
		dcb.fAbortOnError					= 0; 
		dcb.ByteSize							= 8;
		dcb.Parity								= 0;
		dcb.StopBits							= 0;										// one stop bit

		SetCommState(rs232,&dcb);

		// let's see what the time-out functions are...
		GetCommTimeouts(rs232,&cto);
		cto.ReadTotalTimeoutConstant = 2000;							// read timeout of 2000mS
		SetCommTimeouts(rs232,&cto);
		return rs232;
	}
	else
		return 0;		// on error
}

void PrintColour(HANDLE con, char * str, int x, int y, WORD colour)
{
	// print a coloured string at x,y
COORD posn;
	
	posn.X = x;
	posn.Y = y;
	
	SetConsoleTextAttribute(con,colour);
	SetConsoleCursorPosition(con,posn);
	printf("%s",str);
}
void initbosch (HANDLE serial)
{
int q;

	printf("Initialising comms: ");
	// bitbang the  low-baud initialisation
	// send a couple of bits of high/idle, to give the far end time to wake up
	// we use SleepEx() to provide the inter-bit delay
	// we need to send 110000010001
	//														|_ idle state again
	//													 |__ parity
	//								    |_________ 0x10, low bit first, seven bits
	//                   |__________ start bit
	//                 |____________ start idle
	
	// some idle time
	ClearCommBreak(serial);
	SleepEx(200,FALSE);
	printf("1");
	SleepEx(200,FALSE);
	printf("1");
		
	// the start bit
	SetCommBreak(serial);
	SleepEx(200,FALSE);
	printf("0");
	
	// the data, low four bits first
	// it's already zero
	for (q=0; q<4; q++)
	{
		SleepEx(200,FALSE);
		printf("0");
	}
	
	// the high bit 4
	ClearCommBreak(serial);
	SleepEx(200,FALSE);
	printf("1");
	
	// then two remaining zeros
	SetCommBreak(serial);
	for (q=0; q<2; q++)
	{
		SleepEx(200,FALSE);
		printf("0");
	}
		
	// the parity bit - it' a 0
	// still set to zero
	SleepEx(200,FALSE);
	printf("0");
	
	// and finally the stop bit
	ClearCommBreak(serial);
	//SleepEx(100,FALSE);
	//printf("1");
	PurgeComm(serial,PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR);
}

int getboschID()
{
unsigned char idcount = 0;
int error = 0;

	// we don't know how many frames the ID will be hidden in, but it's 30 bytes
	// and we don't really want them at this stage, so we'll just throw them away
	// tsk tsk, how wasteful
	
	while (idcount < 30)
	{
		// the ecu speaks first
		// so we need to listen
		error = framerx();
		
		// we assume it's valid, for now
		idcount+=isoframe[isolength]-3;
	//	printf("count = %d ",idcount);
		
		// we send an ack frame
		//printf("Sending ack\n");
		error = frametx(isoack,0);
		
		printf("*");
	}
	
	// and then let the calling routine deal with the error, if there is one
	// if we got an error, we didn't get sync :(
	// so we need to do the whole damn thing again
	
	// however, if there wasn't an error, we need to wait for a single ack packet
	// before we start talking
	
	error = framerx();					// for now, no testing, just assume that's what it is.
	return error;
}		


// frames and the handling thereof...
// once communication has been established, a master/slave process negotiates
// the actual data transfer.
// To do this a series of frames are used. The initiator of the transfer is the 
// master, the receiver is the slave.
//
// the frame consists of the above structure isoframe, along with a ETX byte
//	length = the following number of bytes in the frame - max 15
//	count  = an incrementing frame counter
//	title  = the type of operation requested or carried out
//				0x01	request reading of ram address
//				0x04	request actuator test
//				0x05	request deletion of error memory
//				0x06	request end of diagnosis
//				0x07	request error memory read
//				0x08	request reading of AD channels
//				0x09	acknowledge
//				0x0a	no acknowledge
//				0x24	transmission of security code
//				0xea	response: security code
//				0xfb	response: AD channels
//				0xfc	response: error memory
//				0xfe	response: ram data
//	info[12]= up to twelve bytes of data or commands
// and then finally the ETX byte 0x03
//
// When the slave receives each byte of the master frame, it shall respond by replying
// with the complement of the byte just received within 40mS, except for the ETX byte which is 
// replied by another ETX byte.
//
// Finally the data requested by the master is returned, with the roles of master and
// slave changing place.
//
// To keep the channel alive, at least an exchange of acknowledge packets must be
// negotiated every second.
//
// Because a single line is used to handle data in both directions, and it is impractical
// to enable and disable the receiver between individual bytes and still guarantee correct
// reception, when a byte is transmitted, it is also received.
//
// Thus, a send_byte routine needs to send the byte requested, receive and discard its own
// echo, receive the complemented response byte, and return a true or false depending 
// on the result.
//
// A frame transmit routine needs to maintain a frame count; create and send the bosch isoframe,
// and return true or false depending on any errors in the return data.
//
// A receive_byte routine needs to store the data and return the complement of the received byte, 
// or ETX as required. (It's rumoured that the ETX is not echoed in either direction - but of course the
// ETX can occur other than at the end of a frame)
//
// A frame receive routine needs to fill an isoframe with incoming data, complementing and
// returning bytes as appropriate; it must also return a true/false

int boschtx (unsigned char ch, int respond)
{
	// transmit a single byte to the line
	// if respond == 1, we expect to receive the complement of the transmitted byte
	// if respond == 0, we expect no response
	// if respond == 2, we expect the transmitted byte returned unchanged
	// wait until an appropriate byte has been returned by the slave
	// or a timeout
	// correct response returns 0;
	// timeout returns 1;
	// incorrect response returns 2;
	
int found;
unsigned char rx;

	WriteFile(rs232,&ch,1,&txcount,NULL);
	// grab the echo byte
	ReadFile(rs232,&rx,1,&rxcount,NULL);
	//printf("%0.2x ",rx);
	
	// now we wait for the ecu to reply - if we haven't sent an ETX
	found = 0;						// assume we will be ok
	
	switch(respond)
	{
		case 0:
			// no reponse expected
			found = 0;
			break;
		
		case 1:
			// expecting complemented return
			found = 1;								// assume we will timeout
			ReadFile(rs232,&rx,1,&rxcount,NULL);
			//printf("%0.2x ", ((~rx) & 0xff));
			if (rxcount != 0)
			{
				// we need to compare it with the outgoing byte
				if (ch != ((~rx) & 0xff))						// if the return wasn't the complement
				{
					found = 2;						// oops, bad return
					//found = 0;
				}
				else
					found = 0;							// good return
			}
			break;
			
		case 2:
			// expecting non-complemented return
			
			found = 1;								// assume we will timeout
			ReadFile(rs232,&rx,1,&rxcount,NULL);
			// we need to compare it with the outgoing byte
			if (ch != rx)							// if the return wasn't the complement
				found = 2;							// oops, bad return
			else
				found = 0;							// good return
			break;
			
		default:
			break;
	}

	// returns 0 for ok, 1 for timeout, 2 for error return
	return found;
}


int boschrx (char * ch, int respond)
{
	// we are listening for a character as part of a frame receive
	// if we receive anything, we have to respond with the complement of
	// the received data
	// in the case of the final ETX, we either respond with the complement or with nothing
	// the docs are not clear
	// if 'respond' = 1, we reply with the complement (we can't just test for ETX as it can occur
	// within the packet).
	// if 'respond' = 0, we do not reply with anything
	// if 'respond' = 2, we reply with the received character
	// after we have transmitted the response we wait for the echo to be received
	// and discard it
	// if we receive no data within two seconds, then the ecu has broken contact
	// so we return 1; if we respond we return 0 (to match tx return)
	
int found = 1;
char rx;

	//printf("*");
	ReadFile(rs232,ch,1,&rxcount,NULL);
	if (rxcount > 0)
	{
		//printf("%0.2x ",*ch);
		// data awaits us, so read it
		// send the response if requested
		found = 0;
		//SleepEx(10, FALSE);							// a short delay
		switch (respond)		
		{
			case 0:		// we return nothing...
				break;
				
			case 1:		// we return the complement
				rx = ~ *ch;
				WriteFile(rs232,&rx,1,&txcount,NULL);
				// grab the echo byte
				ReadFile(rs232,&rx,1,&rxcount,NULL);
				break;
				
			case 2:		// we return the data as received
				WriteFile(rs232,ch,1,&txcount,NULL);
				// grab the echo byte
				ReadFile(rs232,&rx,1,&rxcount,NULL);
				break;
				
			default:
				break;
		}
	}
	
	// now found = 1 if timeout, or 0 if received
	return found;
}


int frametx (unsigned char title, unsigned char length)
{	
	// transmit a single frame to the far end
	// before calling, the frame will have had its info field 
	// filled in by the calling routine
	
int error = 0;
int q;

	//printf("Sending frame - length = %d \n",length);
	// we start by completing the isoframe structure
	isoframe[isolength] = length+3;
	isoframe[isocount] = isoframecount;
	isoframe[isotitle] = title;
	
	for (q=0; q<(length+3); q++)
	{
		// send each byte, watch for errors
		//printf("%0.2x ",isoframe[q]);
		error = boschtx(isoframe[q],1);		// we need to get a complement response from ecu
		if (error == 0) 
		{
			//printf("ok ");
		}

		if (error == 1) 
		{
			//printf("timeout");
		}

		if (error == 2) 
		{
			//printf("Wrong return code error!");
		}
	}
	
	// if we get here without error, we need to send the etx
	//if (error == 0)
		error = boschtx(ETX,0);				// expect non-complement response
		
	// and if we had any errors we need to deal with them, but we let the 
	// calling routine handle it - options are 1 = timeout, 2 = bad response 
	
	return error;
}

int framerx (void)
{
	// receive an isoframe from the ecu
	// we don't know how many bytes until they start coming... we let
	// boschrx look after the echo bytes
	// but we have to watch for the end byte
	
int error = 0;
int q=0;
	
	//printf("framerx\n");
	// clear the frame buffer
	for (q=0; q<16; q++)
	{
		isoframe[q] = 0;
	}
		
	// first byte is always the length
	// returns 1 for timeout error, otherwise zero
	error = boschrx(&isoframe[isolength],1);
	//printf("frame length = %d\n",isoframe[isolength]);
	
	// if we haven't had a timeout error, get the rest of the data
	if (~error)
	{
		// everything except the etx byte
		for (q=0; q<isoframe[isolength]-1; q++)
		{
			error = boschrx(&isoframe[q+isocount], 1);
			if (error)
				break;
		}
		// and then we listen for the etx byte, if no error yet
		if (~error)
		{
		//	printf("waiting for etx\n");
			error = boschrx(&isoframe[q+isocount],0);			// <-- return with no complement
			// now keep the framecount so it can be used by tx
			isoframecount = isoframe[isocount]+1;
			//printf("got etx\n");
		}
	}
	
	// now return with the error value if anything
	// error = 1: timeout

	return error;
}

